/********************************************************************\
|*
|*  Copyright:      Copyright (c) 2008, Altium
|*
|*  Description:    Show how to use the DS1391 device driver with SPI
|*                  bus arbitration being handled by the application.
|*
|*                  Note: since the bus is reserved but never released,
|*                  the firmware of the nanoboard won't be able to read
|*                  either the clock or the touchscreen.
|*
\********************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <time.h>

// Device driver interfaces
#include <devices.h>
#include <drv_spimux.h>
#include <drv_ds1391.h>

static void str2ts( const char * timestring, const char * datestring, struct tm * timestruct );
static void init( void );

// SSAS peripheral pointers
ds1391_t * rtc;
spimux_t * spi;

char * month[] = { "Jan", "Feb", "Mar", "Apr", "May", "Jun", "Jul", "Aug", "Sep", "Oct", "Nov", "Dec" } ;


/**********************************************************************
|*
|*  Function    : main
|*
|*  Parameters  : None
|*
|*  Returns     : None
|*
|*  Description : Main loop - write compilation time to RTC and loop
|*                printing calendar time from that
 */

void main( void )
{
    time_t t;
    struct tm ts;
    int sec;

    init();

    str2ts( __TIME__, __DATE__, &ts );
    t = mktime( &ts );
    ts = * localtime( &t );
    printf( "Date/time = %s", asctime( &ts ));

    ds1391_set_time( rtc, &ts );

    memset( &ts, 0, sizeof( ts ));

    sec = ts.tm_sec;
    for (;;)
    {
        // Loop forever: print date & time once per second
        do
        {
            ds1391_get_time( rtc, &ts );
        } while ( ts.tm_sec == sec );
        sec = ts.tm_sec;
        printf( asctime( &ts ));
    }
}

/**********************************************************************
|*
|*  Function    : str2ts
|*
|*  Parameters  : timestring = ASCII representation of time (hh:mm:ss)
|*                datestring = ASCII representation of date (mon dd yyyy)
|*                timestruct = pointer to time structure to encode date and time in
|*
|*  Returns     : None
|*
|*  Description : Convert a date and time as generated by __FILE__ and __DATE__ macros
|*                to broken down calendartime
 */

static void str2ts( const char * timestring, const char * datestring, struct tm * timestruct )
{
    if ( timestring )
    {
        timestruct->tm_hour = 10 * (timestring[0] - '0') + (timestring[1] - '0');
        timestruct->tm_min = 10 * (timestring[3] - '0') + (timestring[4] - '0');
        timestruct->tm_sec = 10 * (timestring[6] - '0') + (timestring[7] - '0');
    }
    if ( datestring )
    {
        int i;
        char *p = (char *)datestring + 3;
        *p++ = '\0';
        for ( i = 11; i; i-- )
        {
            if ( !strcmp( datestring, month[i] )) break;
        }

        timestruct->tm_mon = i;
        timestruct->tm_mday = *p++ - '0';
        if ( *p != ' ')
        {
            timestruct->tm_mday = timestruct->tm_mday * 10 + (*p++ - '0');

        }
        timestruct->tm_year = 1000 * (p[1] - '0') + 100 * (p[2] - '0') + 10 * (p[3] - '0') + (p[4] - '0') - 1900;
    }

}

/**********************************************************************
|*
|*  Function    : init
|*
|*  Parameters  : None
|*
|*  Returns     : None
|*
|*  Description : Initialize hardware & device drivers
 */

static void init( void )
{
    spi = spimux_open( DRV_SPIMUX_1 );
    while( !spi_get_bus( spi, 6 )) __nop();

    rtc = ds1391_open( DRV_DS1391_1 );

    setbuf( stdout, NULL );
    puts( "File '" __FILE__ "' compiled " __DATE__ ", " __TIME__ );
}

